#= 
----------------------------------------------------------------------------
Model solution.
Entry point.
----------------------------------------------------------------------------
Tristany Armangué-Jubert, Nov 2023
----------------------------------------------------------------------------
=#

### DEPENDENCIES
include("model_functions.jl")
using Parameters, TickTock

# Start timer 
# tick()

# Random seed for replicability
Random.seed!(8)

# Settings 
N_a = 10

# Parametrization 
@with_kw mutable struct Params 
    # Elasticity of the labor supply
    ϵᴸ::Float64 = 3.31785 
    # TFP 
    A::Float64 = 1.0
    # Mass of agents (irrelevant and deprecated)
    Λ::Float64 = 1.0
    # Discount rate 
    β::Float64 = 0.961
    # Death probability
    δ::Float64 = 0.025
    # Revenue elastcity
    xi::Float64 = 0.3332
    # Gumbel shape 
    ν::Float64 = 1.0
    # Mean of productivity
    μ_z::Float64 = 0.0
    # Mean of amenities 
    μ_a::Float64 = 0.0
    
    # Entry cost (entrepreneurship)
    c::Float64 = 1.0
    # Investment cost (HC)
    c_z::Float64 = 1.0
    # Probability of moving up the HC ladder (base) 
    p_n::Float64 = 0.5
    # Probability of moving up the HC ladder (invest)
    p_i::Float64 = 0.55
    # Variance of prod 
    σ_z::Float64 = 0.5
    # Variance of amenities 
    σ_a::Float64 = 0.2
    # Correlation of prod and amenities 
    σ_za::Float64 = 0.0
end

# Pass to struct for initial guess
par = Params()

# Model estimation - Manufacturing sector Netherlands
targets = [59.0222335368171, 0.406902819871902, 30.1278896331787, 0.988350570201874, 0.41840648651123, 1.52645233154297]

# Initial guess - Manufacturing sector Netherlands
x0 = [22.80033798033668, 122.43496994653235, 0.6487165361300524, 0.4988029334027742, 2.133578605916448, 0.8985415338196]

# Estimate parameters 
#estimate(par, 0.3, N_a, targets, x0)
#estimate_2(par, 0.3, N_a, targets, x0)

# Evaluate parameters at the baseline
evaluate(x0, par, N_a, 0.3, targets)

# Evaluate parameters at the deviation of xi
evaluate_jacobian_xi(x0, par, N_a, 0.3, targets)

# # Simulate Netherlands baseline 
counterfactual(x0, par, N_a, 0.3, targets, 3.31785, "baseline_NET", 1e-6, false, true)

# # Simulate Greece counterfactual 
counterfactual(x0, par, N_a, 0.3, targets, 0.61554835, "counter_GRE", 1e-6, false, true)

# Sensitivity of moments to par.c
for eps in [5.0 10.0 12.0 14.0 16.0 18.0 20.0 22.0 26.0 28.0 30.0 32.0 34.0 40.0 45.0 50.0]
   display(eps)
   x1 = [eps, 122.43496994653235, 0.6487165361300524, 0.4988029334027742, 2.133578605916448, 0.8985415338196]    
   sensitivity_c(x1, par, N_a, 0.3, targets)
end

# Sensitivity of moments to par.cz
for eps in [70.0 80.0  90.0 95.0 100.0 105.0 110.0 115.0 120.0 125.0 130.0 135.0 140.0 145.0 150.0 160.0]
   display(eps)
   x1 = [22.80033798033668, eps, 0.6487165361300524, 0.4988029334027742, 2.133578605916448, 0.8985415338196]    
   sensitivity_cz(x1, par, N_a, 0.3, targets)
end


# Sensitivity of moments to par.pi
for eps in [0.525 0.55 0.6 0.625 0.65 0.675 0.7 0.75 0.8 0.85 0.9 0.925 0.95]
   display(eps)
   x1 = [22.80033798033668, 122.43496994653235, eps, 0.4988029334027742, 2.133578605916448, 0.8985415338196]    
   sensitivity_pi(x1, par, N_a, 0.3, targets)
end


# #Sensitivity of moments to par.pn
for eps in [0.16 0.175 0.2 0.275  0.35 0.375 0.425 0.45 0.6 0.65 0.7 0.75 0.8 0.85 0.9 0.95] 
   display(eps)
   x1 = [22.80033798033668, 122.43496994653235, 0.6487165361300524, eps, 2.133578605916448, 0.8985415338196]    
   sensitivity_pn(x1, par, N_a, 0.3, targets)
end

# #Sensitivity of moments to par.sigma_z
for eps = 0.50:+0.1:3.50
    display(eps)
   x1  = [22.80033798033668, 122.43496994653235, 0.6487165361300524, 0.4988029334027742, eps, 0.8985415338196]
    sensitivity_sigmaz(x1, par, N_a, 0.3, targets)
end

# #Sensitivity of moments to par.sigma_a
for eps = 0.05:+0.05:1.4
   display(eps)
   x1 = [22.80033798033668, 122.43496994653235, 0.6487165361300524, 0.4988029334027742, 2.133578605916448, eps]
   sensitivity_sigmaa(x1, par, N_a, 0.3, targets)
end


# #Generic Counterfactual
for eps = 5.5:0.1:5.6
   display(eps)
    ext = "counterfactual_eps_" * replace(string(eps), "."=>"_")
    counterfactual(x0, par, N_a, 0.3, targets, eps, ext, 1e-6, false, false)
end

# Cross-country Counterfactual
df = CSV.read("median_markdown_gdp.csv", DataFrame)
second_column = df[:, 2]  #  or use df.value if you know the column name
for md in second_column
   println("Markdown: ", md)
   eps = 1/(md-1)
   println("Elasticity: ", eps)
   ext = "crosscountry_eps_" * replace(string(eps), "."=>"_")
   counterfactual(x0, par, N_a, 0.3, targets, eps, ext, 1e-9, false, false)
end


# Simulate Greece with counterfactual mechanisms
cρᵉ, cρᶻ, cL_j =  counterfactual(x0, par, N_a, 0.3, targets, 0.61554835, "counter_GRE", 1e-6, false, false)
counterfactual_cf1(x0, par, N_a, 0.3, targets, 3.31785, "baseline_NET_fi", cρᶻ)
counterfactual_cf2(x0, par, N_a, 0.3, targets, 3.31785, "baseline_NET_fe", cρᵉ)
counterfactual_cf3(x0, par, N_a, 0.3, targets, 3.31785, "baseline_NET_fefi", cρᵉ, cρᶻ)



for eps in [0.99]
   display(eps)

   ext = "1_" * replace(string(eps), "."=>"_")
   x1 = [22.80033798033668*eps, 122.43496994653235, 0.6487165361300524, 0.4988029334027742, 2.133578605916448, 0.8985415338196]
   jacobian(x1, par, N_a, 0.3, targets, ext)

   ext = "2_" * replace(string(eps), "."=>"_")
   x1 = [22.80033798033668, 122.43496994653235*eps, 0.6487165361300524, 0.4988029334027742, 2.133578605916448, 0.8985415338196]
   jacobian(x1, par, N_a, 0.3, targets, ext)

   ext = "3_" * replace(string(eps), "."=>"_")
   x1 = [22.80033798033668, 122.43496994653235, 0.6487165361300524*eps, 0.4988029334027742, 2.133578605916448, 0.8985415338196]
   jacobian(x1, par, N_a, 0.3, targets, ext)

   ext = "4_" * replace(string(eps), "."=>"_")
   x1 = [22.80033798033668, 122.43496994653235, 0.6487165361300524, 0.4988029334027742*eps, 2.133578605916448, 0.8985415338196]
   jacobian(x1, par, N_a, 0.3, targets, ext)

   ext = "5_" * replace(string(eps), "."=>"_")
   x1 = [22.80033798033668, 122.43496994653235, 0.6487165361300524, 0.4988029334027742, 2.133578605916448*eps, 0.8985415338196]
   jacobian(x1, par, N_a, 0.3, targets, ext)

   ext = "6_" * replace(string(eps), "."=>"_")
   x1 = [22.80033798033668, 122.43496994653235, 0.6487165361300524, 0.4988029334027742, 2.133578605916448, 0.8985415338196*eps]
   jacobian(x1, par, N_a, 0.3, targets, ext)
end
